### 详细描述

Cobalt 调度管理服务。

---
### 函数文档

#### pthread_getschedparam()

> int pthread_getschedparam(pthread_t thread, int *__restrict__ policy, struct sched_param *__restrict__ param)

获取指定线程的调度策略和参数。

此服务在 `policy` 和 `param` 地址处返回 Cobalt 线程 `thread` 的当前调度策略和调度参数（即优先级）。如果 `thread` 不是 Cobalt 线程的标识符，此服务将回退到常规 POSIX `pthread_getschedparam()` 服务。

**参数**:
- **thread**: 目标线程。
- **policy**: 成功时存储线程调度策略的地址。
- **param**: 成功时存储线程调度参数的地址。

**返回值**:
- 成功时返回 **0**，否则：
    - **ESRCH**: `tid` 无效。

**标签**:
- `thread-unrestricted`

示例代码

```c{filename="app.c"}
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#include <sched.h>
#include <unistd.h>

void* thread_function(void* arg) {
    // 线程工作
    for (int i = 0; i < 3; i++) {
        printf("Thread is running: %d\n", i);
        sleep(1);
    }
    return NULL;
}

int main() {
    pthread_t thread;
    int result;

    // 创建线程
    result = pthread_create(&thread, NULL, thread_function, NULL);
    if (result != 0) {
        perror("pthread_create");
        return EXIT_FAILURE;
    }

    // 获取线程的调度参数
    int policy;
    struct sched_param param;
    result = pthread_getschedparam(thread, &policy, &param);
    if (result != 0) {
        perror("pthread_getschedparam");
        return EXIT_FAILURE;
    }

    // 打印线程的调度策略和优先级
    printf("Thread scheduling policy: ");
    switch (policy) {
        case SCHED_OTHER:
            printf("SCHED_OTHER\n");
            break;
        case SCHED_FIFO:
            printf("SCHED_FIFO\n");
            break;
        case SCHED_RR:
            printf("SCHED_RR\n");
            break;
        default:
            printf("Unknown policy\n");
    }
    printf("Thread priority: %d\n", param.sched_priority);

    // 等待线程结束
    pthread_join(thread, NULL);
    return EXIT_SUCCESS;
}
```

---

#### pthread_getschedparam_ex()

> int pthread_getschedparam_ex(pthread_t thread, int *__restrict__ policy_r, struct sched_param_ex *__restrict__ param_ex)

获取线程的扩展调度策略。

此服务是常规 `pthread_getschedparam()` 服务的扩展版本，还支持主机 Linux 环境中不可用的 Cobalt 特定策略。

**参数**:
- **thread**: 目标线程。
- **policy_r**: 成功时存储 `thread` 的调度策略的地址。
- **param_ex**: 成功时存储 `thread` 的调度参数的地址。

**返回值**:
- 成功时返回 **0**，否则：
    - **ESRCH**: `thread` 无效。

**标签**:
- `thread-unrestricted`

示例代码

```c{filename="app.c"}
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#include <cobalt/sched.h>
#include <unistd.h>

void* thread_function(void* arg) {
    // 线程工作
    for (int i = 0; i < 3; i++) {
        printf("Thread is running: %d\n", i);
        sleep(1);
    }
    return NULL;
}

int main() {
    pthread_t thread;
    int result;

    // 创建线程
    result = pthread_create(&thread, NULL, thread_function, NULL);
    if (result != 0) {
        perror("pthread_create");
        return EXIT_FAILURE;
    }

    // 获取线程的调度参数
    int policy;
    struct sched_param_ex param;
    result = pthread_getschedparam_ex(thread, &policy, &param);
    if (result != 0) {
        perror("pthread_getschedparam");
        return EXIT_FAILURE;
    }

    // 打印线程的调度策略和优先级
    printf("Thread scheduling policy: ");
    switch (policy) {
        case SCHED_OTHER:
            printf("SCHED_OTHER\n");
            break;
        case SCHED_FIFO:
            printf("SCHED_FIFO\n");
            break;
        case SCHED_RR:
            printf("SCHED_RR\n");
            break;
        default:
            printf("Unknown policy\n");
    }
    printf("Thread priority: %d\n", param.sched_priority);

    // 等待线程结束
    pthread_join(thread, NULL);
    return EXIT_SUCCESS;
}
```

---

#### pthread_setschedparam()

> int pthread_setschedparam(pthread_t thread, int policy, const struct sched_param *param)

设置指定线程的调度策略和参数。

此服务将由 `thread` 标识的 Cobalt 线程的调度策略设置为 `policy`，并将其调度参数（即优先级）设置为 `param` 指向的值。

如果传递 `pthread_self()`，此服务将当前线程转换为 Cobalt 线程。如果 `thread` 不是 Cobalt 线程的标识符，此服务将回退到常规的 `pthread_setschedparam()` 服务。

**参数**:
- **thread**: 目标 Cobalt 线程。
- **policy**: 调度策略，取值为 `SCHED_FIFO`、`SCHED_RR` 或 `SCHED_OTHER`。
- **param**: 调度参数的地址。

**返回值**:
- 成功时返回 **0**，否则：
    - **ESRCH**: `thread` 无效。
    - **EINVAL**: `policy` 或 `param->sched_priority` 无效。
    - **EAGAIN**: 系统堆内存不足，请增加 `CONFIG_XENO_OPT_SYS_HEAPSZ`。
    - **EFAULT**: `param` 是无效地址。

**另请参见**:
- `pthread_create()`，`pthread_setschedparam_ex()`。

**标签**:
- `thread-unrestricted`，`switch-secondary`，`switch-primary`

示例代码

```c{filename="app.c"}
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sched.h>
#include <errno.h>
#include <unistd.h>

void* thread_function(void* arg) {
    // 线程工作内容
    printf("Thread is running\n");
    sleep(1);
    printf("Thread finished\n");
    return NULL;
}

int main() {
    pthread_t thread_id;
    struct sched_param param;
    int policy = SCHED_FIFO; // 调度策略
    int priority = 50; // 调度优先级

    // 设置线程的调度参数
    param.sched_priority = priority;

    // 创建线程
    if (pthread_create(&thread_id, NULL, thread_function, NULL) != 0) {
        perror("pthread_create");
        exit(EXIT_FAILURE);
    }

    // 设置线程调度策略和参数
    if (pthread_setschedparam(thread_id, policy, &param) != 0) {
        perror("pthread_setschedparam");
        exit(EXIT_FAILURE);
    }

    printf("Thread created with SCHED_FIFO policy and priority %d\n", priority);

    // 等待线程结束
    pthread_join(thread_id, NULL);

    return 0;
}
```

---

### pthread_setschedparam_ex()

> int pthread_setschedparam_ex(pthread_t thread, int policy, const struct sched_param_ex *param_ex)

设置线程的扩展调度策略。

此服务是常规 `pthread_setschedparam()` 服务的扩展版本，支持主机 Linux 环境中不可用的 Cobalt 特定调度策略。

此服务将 Cobalt 线程 `thread` 的调度策略设置为 `policy` 的值，并将调度参数（例如其优先级）设置为 `param_ex` 指向的值。

如果 `thread` 不是 Cobalt 线程的标识符，此操作将回退到常规的 `pthread_setschedparam()` 服务。

**参数**:
- **thread**: 目标 Cobalt 线程。
- **policy**: 调度策略，取值为 `SCHED_WEAK`、`SCHED_FIFO`、`SCHED_COBALT`、`SCHED_RR`、`SCHED_SPORADIC`、`SCHED_TP`、`SCHED_QUOTA` 或 `SCHED_NORMAL`。
- **param_ex**: 调度参数的地址。作为特殊情况，负的 `sched_priority` 值被解释为 `policy` 中的 `SCHED_WEAK`，使用该参数的绝对值作为弱优先级级别。

当启用 `CONFIG_XENO_OPT_SCHED_WEAK` 时，`SCHED_WEAK` 展示 [0..99] 范围内的优先级级别（包括边界值）。否则，对于 `SCHED_WEAK` 策略，`sched_priority` 必须为零。

**返回值**:
- 成功时返回 **0**，否则：
    - **ESRCH**: `thread` 无效。
    - **EINVAL**: `policy` 或 `param_ex->sched_priority` 无效。
    - **EAGAIN**: 系统堆内存不足，请增加 `CONFIG_XENO_OPT_SYS_HEAPSZ`。
    - **EFAULT**: `param_ex` 是无效地址。
    - **EPERM**: 调用进程没有超级用户权限。

**另请参见**:
- `pthread_create()`，`pthread_setschedparam()`。

**标签**:
- `thread-unrestricted`，`switch-secondary`，`switch-primary`。

由 `pthread_setschedparam()` 引用。

示例代码

```c{filename="app.c"}
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <cobalt/sched.h>
#include <errno.h>
#include <unistd.h>

void* thread_function(void* arg) {
    // 线程工作内容
    printf("Thread is running\n");
    sleep(1);
    printf("Thread finished\n");
    return NULL;
}

int main() {
    pthread_t thread_id;
    struct sched_param_ex param;
    int policy = SCHED_FIFO; // 调度策略
    int priority = 50; // 调度优先级

    // 设置线程的调度参数
    param.sched_priority = priority;

    // 创建线程
    if (pthread_create(&thread_id, NULL, thread_function, NULL) != 0) {
        perror("pthread_create");
        exit(EXIT_FAILURE);
    }

    // 设置线程调度策略和参数
    if (pthread_setschedparam_ex(thread_id, policy, &param) != 0) {
        perror("pthread_setschedparam");
        exit(EXIT_FAILURE);
    }

    printf("Thread created with SCHED_FIFO policy and priority %d\n", priority);

    // 等待线程结束
    pthread_join(thread_id, NULL);

    return 0;
}
```

---

#### pthread_yield()

> int pthread_yield(void)

让出处理器。

此函数将当前线程移动到其优先级组的末尾，允许其他具有相同优先级的线程运行。

**返回值**:
- 成功时返回 **0**。

**标签**:
- `thread-unrestricted`，`switch-primary`

引用 `sched_yield()`。

示例代码

```c{filename="app.c"}
#include <pthread.h>
#include <stdio.h>
#include <unistd.h>

void* thread_function(void* arg) {
    printf("Thread is running and will yield...\n");
    pthread_yield(); // 放弃当前线程的时间片
    printf("Thread continues after yielding.\n");
    return NULL;
}

int main() {
    pthread_t thread_id;

    // 创建线程
    if (pthread_create(&thread_id, NULL, thread_function, NULL) != 0) {
        perror("pthread_create");
        return 1;
    }

    printf("Main thread is running.\n");
    sleep(1); // 给线程一点时间运行

    // 等待线程结束
    pthread_join(thread_id, NULL);
    printf("Main thread finished.\n");

    return 0;
}
```

---